import Foundation
import CoreData

/// Database service for managing historical prediction data
/// Provides efficient querying, pagination, and trend calculations
class DatabaseService: ObservableObject {
    static let shared = DatabaseService()
    
    private let coreDataStack = CoreDataStack.shared
    private let dateFormatter: DateFormatter = {
        let formatter = DateFormatter()
        formatter.dateStyle = .short
        return formatter
    }()
    
    private init() {}
    
    // MARK: - Save Operations
    
    /// Saves a prediction record to Core Data
    /// - Parameters:
    ///   - lifeExpectancyYears: The predicted life expectancy in years
    ///   - predictedDeathDate: The calculated predicted death date
    ///   - healthMetrics: Dictionary of health metrics used for prediction
    ///   - confidence: Confidence score for the prediction (0.0 - 1.0)
    /// - Returns: Success status of save operation
    @discardableResult
    func savePrediction(
        lifeExpectancyYears: Double,
        predictedDeathDate: Date,
        healthMetrics: [String: Any],
        confidence: Double = 0.0
    ) -> Bool {
        let context = coreDataStack.backgroundContext
        var success = false
        
        context.performAndWait {
            // Calculate change from previous prediction
            let changeFromPrevious = calculateChangeFromPrevious(
                newLifeExpectancy: lifeExpectancyYears,
                context: context
            )
            
            // Create new prediction record
            let predictionRecord = PredictionRecord(context: context)
            predictionRecord.date = Date()
            predictionRecord.lifeExpectancyYears = lifeExpectancyYears
            predictionRecord.predictedDeathDate = predictedDeathDate
            predictionRecord.changeFromPrevious = changeFromPrevious
            predictionRecord.confidence = confidence
            
            // Extract health metrics
            predictionRecord.restingHeartRate = healthMetrics["restingHeartRate"] as? Double ?? 0.0
            predictionRecord.hrv = healthMetrics["hrv"] as? Double ?? 0.0
            predictionRecord.vo2Max = healthMetrics["vo2Max"] as? Double ?? 0.0
            predictionRecord.bmi = healthMetrics["bmi"] as? Double ?? 0.0
            predictionRecord.stepCount = Int32(healthMetrics["stepCount"] as? Int ?? 0)
            predictionRecord.sleepHours = healthMetrics["sleepHours"] as? Double ?? 0.0
            
            do {
                try context.save()
                success = true
                print("✅ Saved prediction record: \(lifeExpectancyYears) years")
            } catch {
                print("❌ Failed to save prediction record: \(error)")
            }
        }
        
        return success
    }
    
    // MARK: - Fetch Operations
    
    /// Fetches prediction history as PredictionHistoryEntry for UI consumption
    /// - Parameter timeframe: The timeframe to filter by
    /// - Returns: Array of prediction history entries
    func fetchPredictionHistory(timeframe: HistoricalTimeframe) async throws -> [PredictionHistoryEntry] {
        return await withUnsafeContinuation { continuation in
            coreDataStack.backgroundContext.perform {
                let fetchRequest: NSFetchRequest<PredictionRecord> = PredictionRecord.fetchRequest()
                
                // Configure timeframe filtering
                let startDate = Calendar.current.date(byAdding: .day, value: -timeframe.days, to: Date()) ?? Date()
                fetchRequest.predicate = NSPredicate(format: "date >= %@", startDate as NSDate)
                
                // Configure sorting (oldest first for trend analysis)
                fetchRequest.sortDescriptors = [
                    NSSortDescriptor(key: "date", ascending: true)
                ]
                
                do {
                    let records = try self.coreDataStack.backgroundContext.fetch(fetchRequest)
                    let historyEntries = records.map { record in
                        PredictionHistoryEntry(
                            timestamp: record.date ?? Date(),
                            lifeExpectancy: record.lifeExpectancyYears,
                            avgHeartRate: record.restingHeartRate > 0 ? record.restingHeartRate : nil,
                            dailySteps: record.stepCount > 0 ? Double(record.stepCount) : nil,
                            sleepHours: record.sleepHours > 0 ? record.sleepHours : nil,
                            exerciseMinutes: nil, // Would need to add this field to PredictionRecord
                            contributingFactors: [] // Would need to implement this
                        )
                    }
                    continuation.resume(returning: historyEntries)
                } catch {
                    print("❌ Failed to fetch prediction history: \(error)")
                    continuation.resume(returning: [])
                }
            }
        }
    }
    
    /// Fetches prediction history with pagination and filtering (legacy method)
    /// - Parameters:
    ///   - timeframe: The timeframe to filter by (day, week, month, year)
    ///   - limit: Maximum number of records to return
    ///   - offset: Number of records to skip (for pagination)
    /// - Returns: Array of prediction records
    func fetchPredictionHistory(
        timeframe: PredictionTimeframe = .all,
        limit: Int = 100,
        offset: Int = 0
    ) -> [PredictionRecord] {
        let context = coreDataStack.viewContext
        let fetchRequest: NSFetchRequest<PredictionRecord> = PredictionRecord.fetchRequest()
        
        // Configure timeframe filtering
        if timeframe != .all {
            let startDate = timeframe.startDate()
            fetchRequest.predicate = NSPredicate(format: "date >= %@", startDate as NSDate)
        }
        
        // Configure sorting (newest first)
        fetchRequest.sortDescriptors = [
            NSSortDescriptor(key: "date", ascending: false)
        ]
        
        // Configure pagination
        fetchRequest.fetchLimit = limit
        fetchRequest.fetchOffset = offset
        
        do {
            let records = try context.fetch(fetchRequest)
            print("📊 Fetched \(records.count) prediction records for timeframe: \(timeframe)")
            return records
        } catch {
            print("❌ Failed to fetch prediction history: \(error)")
            return []
        }
    }
    
    /// Fetches the most recent prediction record
    /// - Returns: The most recent prediction record, or nil if none exists
    func fetchLatestPrediction() -> PredictionRecord? {
        let context = coreDataStack.viewContext
        let fetchRequest: NSFetchRequest<PredictionRecord> = PredictionRecord.fetchRequest()
        
        fetchRequest.sortDescriptors = [
            NSSortDescriptor(key: "date", ascending: false)
        ]
        fetchRequest.fetchLimit = 1
        
        do {
            let records = try context.fetch(fetchRequest)
            return records.first
        } catch {
            print("❌ Failed to fetch latest prediction: \(error)")
            return nil
        }
    }
    
    /// Fetches records for a specific date range
    /// - Parameters:
    ///   - startDate: Start of date range
    ///   - endDate: End of date range
    /// - Returns: Array of prediction records within the date range
    func fetchPredictions(from startDate: Date, to endDate: Date) -> [PredictionRecord] {
        let context = coreDataStack.viewContext
        let fetchRequest: NSFetchRequest<PredictionRecord> = PredictionRecord.fetchRequest()
        
        fetchRequest.predicate = NSPredicate(
            format: "date >= %@ AND date <= %@",
            startDate as NSDate,
            endDate as NSDate
        )
        
        fetchRequest.sortDescriptors = [
            NSSortDescriptor(key: "date", ascending: true)
        ]
        
        do {
            return try context.fetch(fetchRequest)
        } catch {
            print("❌ Failed to fetch predictions for date range: \(error)")
            return []
        }
    }
    
    // MARK: - Trend Calculations
    
    /// Calculates trend data for the specified timeframe
    /// - Parameter timeframe: The timeframe to analyze
    /// - Returns: Comprehensive trend analysis
    func calculateTrend(for timeframe: PredictionTimeframe) -> PredictionTrend {
        let records = fetchPredictionHistory(timeframe: timeframe, limit: 1000)
        
        guard !records.isEmpty else {
            return PredictionTrend.empty
        }
        
        let lifeExpectancies = records.map { $0.lifeExpectancyYears }
        
        let average = lifeExpectancies.reduce(0, +) / Double(lifeExpectancies.count)
        let maximum = lifeExpectancies.max() ?? 0
        let minimum = lifeExpectancies.min() ?? 0
        
        // Calculate trend direction
        let trendDirection = calculateTrendDirection(from: records)
        
        // Find best and worst days
        let bestDay = records.max { $0.lifeExpectancyYears < $1.lifeExpectancyYears }
        let worstDay = records.min { $0.lifeExpectancyYears < $1.lifeExpectancyYears }
        
        // Calculate variance and standard deviation
        let variance = calculateVariance(values: lifeExpectancies, mean: average)
        let standardDeviation = sqrt(variance)
        
        return PredictionTrend(
            timeframe: timeframe,
            recordCount: records.count,
            averageLifeExpectancy: average,
            maximumLifeExpectancy: maximum,
            minimumLifeExpectancy: minimum,
            trendDirection: trendDirection,
            variance: variance,
            standardDeviation: standardDeviation,
            bestDay: bestDay?.date,
            worstDay: worstDay?.date,
            bestDayValue: bestDay?.lifeExpectancyYears ?? 0,
            worstDayValue: worstDay?.lifeExpectancyYears ?? 0
        )
    }
    
    /// Calculates correlation between health metrics and life expectancy changes
    /// - Parameter timeframe: The timeframe to analyze
    /// - Returns: Dictionary of correlations for each health metric
    func calculateHealthMetricCorrelations(for timeframe: PredictionTimeframe) -> [String: Double] {
        let records = fetchPredictionHistory(timeframe: timeframe, limit: 1000)
        
        guard records.count > 2 else { return [:] }
        
        var correlations: [String: Double] = [:]
        
        // Calculate correlations for each health metric
        let metrics = ["restingHeartRate", "hrv", "vo2Max", "bmi", "stepCount", "sleepHours"]
        
        for metric in metrics {
            let metricValues = records.compactMap { record in
                getValue(for: metric, from: record)
            }
            
            let lifeExpectancies = records.map { $0.lifeExpectancyYears }
            
            if metricValues.count == lifeExpectancies.count && metricValues.count > 1 {
                correlations[metric] = calculateCorrelation(
                    x: metricValues,
                    y: lifeExpectancies
                )
            }
        }
        
        return correlations
    }
    
    // MARK: - Utility Functions
    
    /// Deletes old records beyond the specified limit to manage storage
    /// - Parameter keepRecords: Number of most recent records to keep
    func cleanupOldRecords(keepRecords: Int = 10000) {
        coreDataStack.saveInBackground { context in
            let fetchRequest: NSFetchRequest<PredictionRecord> = PredictionRecord.fetchRequest()
            fetchRequest.sortDescriptors = [
                NSSortDescriptor(key: "date", ascending: false)
            ]
            
            do {
                let allRecords = try context.fetch(fetchRequest)
                
                if allRecords.count > keepRecords {
                    let recordsToDelete = Array(allRecords.dropFirst(keepRecords))
                    
                    for record in recordsToDelete {
                        context.delete(record)
                    }
                    
                    print("🧹 Cleaned up \(recordsToDelete.count) old prediction records")
                }
            } catch {
                print("❌ Failed to cleanup old records: \(error)")
            }
        }
    }
    
    // MARK: - Private Helper Methods
    
    private func calculateChangeFromPrevious(
        newLifeExpectancy: Double,
        context: NSManagedObjectContext
    ) -> Double {
        let fetchRequest: NSFetchRequest<PredictionRecord> = PredictionRecord.fetchRequest()
        fetchRequest.sortDescriptors = [NSSortDescriptor(key: "date", ascending: false)]
        fetchRequest.fetchLimit = 1
        
        do {
            let previousRecords = try context.fetch(fetchRequest)
            if let previousRecord = previousRecords.first {
                return newLifeExpectancy - previousRecord.lifeExpectancyYears
            }
        } catch {
            print("❌ Failed to fetch previous record: \(error)")
        }
        
        return 0.0
    }
    
    private func calculateTrendDirection(from records: [PredictionRecord]) -> TrendDirection {
        guard records.count >= 2 else { return .stable }
        
        let sortedRecords = records.sorted { $0.date! < $1.date! }
        let firstHalf = sortedRecords.prefix(sortedRecords.count / 2)
        let secondHalf = sortedRecords.suffix(sortedRecords.count / 2)
        
        let firstHalfAvg = firstHalf.map { $0.lifeExpectancyYears }.reduce(0, +) / Double(firstHalf.count)
        let secondHalfAvg = secondHalf.map { $0.lifeExpectancyYears }.reduce(0, +) / Double(secondHalf.count)
        
        let difference = secondHalfAvg - firstHalfAvg
        
        if difference > 0.1 {
            return .improving
        } else if difference < -0.1 {
            return .declining
        } else {
            return .stable
        }
    }
    
    private func calculateVariance(values: [Double], mean: Double) -> Double {
        guard !values.isEmpty else { return 0.0 }
        
        let squaredDifferences = values.map { pow($0 - mean, 2) }
        return squaredDifferences.reduce(0, +) / Double(values.count)
    }
    
    private func calculateCorrelation(x: [Double], y: [Double]) -> Double {
        guard x.count == y.count && x.count > 1 else { return 0.0 }
        
        let n = Double(x.count)
        let sumX = x.reduce(0, +)
        let sumY = y.reduce(0, +)
        let sumXY = zip(x, y).map(*).reduce(0, +)
        let sumX2 = x.map { $0 * $0 }.reduce(0, +)
        let sumY2 = y.map { $0 * $0 }.reduce(0, +)
        
        let numerator = (n * sumXY) - (sumX * sumY)
        let denominator = sqrt((n * sumX2 - sumX * sumX) * (n * sumY2 - sumY * sumY))
        
        guard denominator != 0 else { return 0.0 }
        
        return numerator / denominator
    }
    
    private func getValue(for metric: String, from record: PredictionRecord) -> Double? {
        switch metric {
        case "restingHeartRate":
            return record.restingHeartRate > 0 ? record.restingHeartRate : nil
        case "hrv":
            return record.hrv > 0 ? record.hrv : nil
        case "vo2Max":
            return record.vo2Max > 0 ? record.vo2Max : nil
        case "bmi":
            return record.bmi > 0 ? record.bmi : nil
        case "stepCount":
            return Double(record.stepCount)
        case "sleepHours":
            return record.sleepHours > 0 ? record.sleepHours : nil
        default:
            return nil
        }
    }
}

// MARK: - Supporting Types

enum PredictionTimeframe: String, CaseIterable {
    case day = "24h"
    case week = "7d"
    case month = "30d"
    case year = "365d"
    case all = "all"
    
    func startDate() -> Date {
        let calendar = Calendar.current
        let now = Date()
        
        switch self {
        case .day:
            return calendar.date(byAdding: .day, value: -1, to: now) ?? now
        case .week:
            return calendar.date(byAdding: .day, value: -7, to: now) ?? now
        case .month:
            return calendar.date(byAdding: .day, value: -30, to: now) ?? now
        case .year:
            return calendar.date(byAdding: .day, value: -365, to: now) ?? now
        case .all:
            return Date.distantPast
        }
    }
    
    var displayName: String {
        switch self {
        case .day: return "Today"
        case .week: return "This Week"
        case .month: return "This Month" 
        case .year: return "This Year"
        case .all: return "All Time"
        }
    }
}


struct PredictionTrend {
    let timeframe: PredictionTimeframe
    let recordCount: Int
    let averageLifeExpectancy: Double
    let maximumLifeExpectancy: Double
    let minimumLifeExpectancy: Double
    let trendDirection: TrendDirection
    let variance: Double
    let standardDeviation: Double
    let bestDay: Date?
    let worstDay: Date?
    let bestDayValue: Double
    let worstDayValue: Double
    
    static let empty = PredictionTrend(
        timeframe: .all,
        recordCount: 0,
        averageLifeExpectancy: 0,
        maximumLifeExpectancy: 0,
        minimumLifeExpectancy: 0,
        trendDirection: .stable,
        variance: 0,
        standardDeviation: 0,
        bestDay: nil,
        worstDay: nil,
        bestDayValue: 0,
        worstDayValue: 0
    )
}